<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Providers\RouteServiceProvider;
use App\Models\User;
use Illuminate\Foundation\Auth\RegistersUsers;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Faker\Factory as Faker;
use Illuminate\Support\Str;
use App\Rules\ValidReferralCode;
use Illuminate\Validation\Rule;


class RegisterController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Register Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users as well as their
    | validation and creation. By default this controller uses a trait to
    | provide this functionality without requiring any additional code.
    |
    */

    use RegistersUsers;

    /**
     * Where to redirect users after registration.
     *
     * @var string
     */
    protected $redirectTo = RouteServiceProvider::HOME;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest');
    }

    /**
     * Get a validator for an incoming registration request.
     *
     * @param  array  $data
     * @return \Illuminate\Contracts\Validation\Validator
     */
    protected function validator(array $data)
    {
        return Validator::make($data, [
            'username' => [
                'required',
                'string',
                'min:3',
                'max:20',
                'regex:/^[a-zA-Z0-9_]+$/u', // Only allow alphanumeric characters and underscores
                Rule::unique('users', 'username'), // Ensure username is unique in the users table
            ],
            'mobile_number' => ['required', 'numeric', Rule::unique('users', 'mobile_number')],
            'referral_code' => ['required', new ValidReferralCode],
            'redemption_pin' => ['required', 'numeric','digits:4'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
        ]);
    }

    function generateReferralCode($length = 6)
    {
        $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $referralCode = '';

        for ($i = 0; $i < $length; $i++) {
            $referralCode .= $characters[rand(0, strlen($characters) - 1)];
        }

        return strtoupper($referralCode);
    }

    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array  $data
     * @return \App\Models\User
     */
    protected function create(array $data)
    {
        $faker = Faker::create();
        $dummyName = $faker->name;
        $email = $faker->email;

        $referralCode = $this->generateReferralCode(); // You can adjust the length as needed

        // Ensure the referral code is unique
        while (User::where('referral_code', $referralCode)->exists()) {
            $referralCode = $this->generateReferralCode();
        }

        $check_referral =  User::where('referral_code', $data['referral_code'])->first();


        $user =  User::create([
            'name' => $dummyName,
            'email' => $email,
            'type' => 'User',
            'username' => $data['username'],
            'mobile_number' => $data['mobile_number'],
            'redemption_pin' => $data['redemption_pin'],
            'referral_code' => $referralCode,
            'referral_by' => $check_referral->id,
            'password' => Hash::make($data['password']),
        ]);

        $user->assignRole('User');
        return  $user;
    }
}
