<?php

namespace App\DataTables;

use App\Facades\UtilityFacades;
use App\Models\RatingHistory;
use App\Models\User;
use Illuminate\Http\Request;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class OrdersDataTable extends DataTable
{
    /**
     * Build DataTable class.
     *
     * @param mixed $query Results from query() method.
     * @return \Yajra\DataTables\DataTableAbstract
     */
    public function dataTable($query)
    {
        return datatables()
            ->eloquent($query)
             ->addColumn('username', function (RatingHistory $user) {
                return $user->userDetails->username;
             })
            ->addColumn('finish_start', function (RatingHistory $user) {
                return $user->userDetails->visited_journey . '/40';
            })

            ->addColumn('product_name', function (RatingHistory $user) {
                return $user->journeyDetails->name;
            })
            ->addColumn('num_of_transaction', function (RatingHistory $user) {
                return $user->userDetails->visited_journey;
            })
            ->addColumn('price', function (RatingHistory $user) {
                return round($user->journeyDetails->price * $user->journeyDetails->qty,2);
            })
            
            ->editColumn('created_at', function (RatingHistory $user) {
                return UtilityFacades::date_time_format($user->created_at);
            })
            ->addColumn('action', function (RatingHistory $user) {
                return view('orders.action', compact('user'));
            })

            ->rawColumns(['role', 'action']);
       
    }

    /**
     * Get query source of dataTable.
     *
     * @param \App\User $model
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query(RatingHistory $model, Request $request)
    {

        $users =  $model->newQuery();
        if ($request->username) {
            $users->where('username', 'like', '%' . $request->username . '%');
        }
        if ($request->mobile_number) {
            $users->where('mobile_number', 'like', '%' . $request->mobile_number . '%');
        }
        if ($request->r_date) {
            // dd($request->r_date);
            $users->whereDate('created_at', '=', $request->r_date);
        }
        return $users;
    }

    /**
     * Optional method if you want to use html builder.
     *
     * @return \Yajra\DataTables\Html\Builder
     */
    public function html()
    {
        return $this->builder()
            ->setTableId('users-table')
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->orderBy(1)
            ->language([
                "paginate" => [
                    "next" => '<i class="fas fa-angle-right"></i>',
                    "previous" => '<i class="fas fa-angle-left"></i>'
                ]
            ])
            ->parameters([
                "scrollX" => true
            ]);
    }

    /**
     * Get columns.
     *
     * @return array
     */
    protected function getColumns()
    {
        return [
            
            Column::make('id'),
            Column::make('username'),
            Column::make('finish_start'),
            Column::make('product_name'),
            Column::make('price'),
            Column::make('num_of_transaction'),
            Column::make('commission'),
            Column::make('created_at'),
            
            Column::computed('action')
                ->exportable(false)
                ->printable(false)
                ->width(200)
                ->addClass('text-center'),
        ];
    }

    /**
     * Get filename for export.
     *
     * @return string
     */
    // protected function filename()
    // {
    //     return 'Users_' . date('YmdHis');
    // }
}
